<?php

use Livewire\Volt\Component;
use Livewire\WithFileUploads;
use App\Models\FastEventsWelcome;
use App\Models\SRO_VT_SHARD\Char;
use App\Models\SRO_VT_PROXY\_Players;
use App\Models\SRO_VT_SHARD\_User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

new class extends Component {
    use WithFileUploads;

    public $selectedPlayerType = null;
    public $selectedCharacter = null;
    public $oldCharacterName = '';
    public $oldCharacterProofs = [];
    public $donationAmount = 0;
    public $selectedCurrency = 'USD';
    public $donationProofs = [];
    public $oldCharacterImages = [];
    public $calculatedBalance = 0;
    public $userApplication = null;
    public $userCharacters = [];
    public $isSubmitting = false;
    public $isClaiming = false;

    public function mount()
    {
        $this->loadUserData();
        $this->checkForRejectionNotification();
    }

    public function checkForRejectionNotification()
    {
        // Check for any recently rejected application (within last 10 minutes)
        $recentRejection = FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->where('reviewed_at', '>=', now()->subMinutes(10))
            ->latest()
            ->first();

        if ($recentRejection) {
            // Show rejection notification if recently rejected
            $this->dispatch('show-rejection-notification', [
                'applicationId' => $recentRejection->id,
                'rejectionReason' => $recentRejection->rejection_reason ?? 'No specific reason provided',
                'reviewedAt' => $recentRejection->reviewed_at->format('M d, Y H:i'),
                'playerType' => ucfirst($recentRejection->player_type) . ' Player',
                'rewardAmount' => $recentRejection->reward_amount
            ]);
        }
    }

    public function loadUserData()
    {
        $user = Auth::user();

        // Get the latest application for the user (to show current status)
        // This allows users to have multiple applications if previous ones were rejected
        $this->userApplication = FastEventsWelcome::where('JID', $user->JID)
            ->whereIn('status', ['pending', 'approved', 'claimed']) // Don't include rejected ones
            ->latest('created_at')
            ->first();

        // Load user characters: Get CharIDs from _User table first, then get character details
        $userCharIds = _User::where('UserJID', $user->JID)->pluck('CharID');
        $this->userCharacters = Char::whereIn('CharID', $userCharIds)
            ->select('CharName16', 'CharID', 'RefObjID', 'CurLevel')
            ->orderBy('CharName16')
            ->get()
            ->map(function($char) {
                // Check if character is active in _Players table
                $player = _Players::where('CharID', $char->CharID)->first();
                $isActive = $player && $player->cur_status == true;

                return [
                    'CharID' => $char->CharID,
                    'CharName16' => $char->CharName16,
                    'RefObjID' => $char->RefObjID,
                    'CurLevel' => $char->CurLevel,
                    'avatar' => Char::getCharAvatarGifByCharname($char->CharName16),
                    'isActive' => $isActive
                ];
            });

        // Check for rejection notification after loading data
        $this->checkForRejectionNotification();
    }

    public function getRejectedApplications()
    {
        return FastEventsWelcome::where('JID', Auth::user()->JID)
            ->where('status', 'rejected')
            ->latest('reviewed_at')
            ->limit(3) // Show only last 3 rejected applications
            ->get();
    }

    public function selectPlayerType($type)
    {
        $this->selectedPlayerType = $type;
        $this->calculateReward();
    }

    public function selectCharacter($characterId)
    {
        // Check if this character belongs to the user by checking through _User table
        $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

        if ($userCharIds->contains($characterId)) {
            // For all player types, check if character has cur_status = true in _Players table
            $player = _Players::where('CharID', $characterId)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            $this->selectedCharacter = $characterId;
        }
    }

    public function updatedDonationAmount()
    {
        $this->calculateReward();
    }

    public function updatedSelectedCurrency()
    {
        $this->calculateReward();
    }

    public function calculateReward()
    {
        if ($this->selectedPlayerType === 'donate' && $this->donationAmount > 0) {
            $rate = $this->selectedCurrency === 'USD' ? 40 : 1;
            $this->calculatedBalance = floor(($this->donationAmount * $rate) * 0.5);
        } else {
            $this->calculatedBalance = 0;
        }
    }

    public function submitApplication()
    {
        $this->isSubmitting = true;

        try {
            // Validate based on player type
            $rules = [
                'selectedPlayerType' => 'required|in:new,old,donate',
                'selectedCharacter' => 'required|numeric',
            ];

            // Only validate old character specific fields for old players
            if ($this->selectedPlayerType === 'old') {
                $rules['oldCharacterName'] = 'required|string|max:255';
                $rules['oldCharacterProofs'] = 'required|array|min:1';
                $rules['oldCharacterProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            // Only validate donation specific fields for donates
            if ($this->selectedPlayerType === 'donate') {
                $rules['donationAmount'] = 'required|numeric|min:1';
                $rules['selectedCurrency'] = 'required|in:USD,EGP';
                $rules['donationProofs'] = 'required|array|min:1';
                $rules['donationProofs.*'] = 'image|mimes:jpeg,png,jpg,gif|max:5120';
            }

            $this->validate($rules);

            // Check character ownership through _User table
            $userCharIds = _User::where('UserJID', Auth::user()->JID)->pluck('CharID');

            if (!$userCharIds->contains($this->selectedCharacter)) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found or does not belong to you.',
                    'duration' => 2500,
                ]);
                return;
            }

            // Get character details
            $character = Char::where('CharID', $this->selectedCharacter)->first();
            if (!$character) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character not found in database.',
                    'duration' => 2500,
                ]);
                return;
            }

            // For all player types, check if character has been played (cur_status = true)
            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            if (!$player || !$player->cur_status) {
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Character must be active (played at least once) for all applications.',
                    'duration' => 2500,
                ]);
                return;
            }

            // Store uploaded files
            $oldCharacterProofPaths = [];
            $donationProofPaths = [];
            $oldCharacterImagePaths = [];

            if ($this->selectedPlayerType === 'old' && !empty($this->oldCharacterProofs)) {
                foreach ($this->oldCharacterProofs as $file) {
                    $path = $file->store('welcome-event/old-character-proofs', 'public');
                    $oldCharacterProofPaths[] = $path;
                }
            }

            if ($this->selectedPlayerType === 'donate' && !empty($this->donationProofs)) {
                foreach ($this->donationProofs as $file) {
                    $path = $file->store('welcome-event/donation-proofs', 'public');
                    $donationProofPaths[] = $path;
                }
            }

            if (!empty($this->oldCharacterImages)) {
                foreach ($this->oldCharacterImages as $file) {
                    $path = $file->store('welcome-event/old-character-images', 'public');
                    $oldCharacterImagePaths[] = $path;
                }
            }

            // Calculate reward amount
            $rewardAmount = $this->getRewardAmount();

            // Get HWID from _Players table
            $player = _Players::where('CharID', $this->selectedCharacter)->first();
            $hwid = $player ? $player->Serial : '';

            // Create application
            FastEventsWelcome::create([
                'JID' => Auth::user()->JID,
                'HWID' => $hwid,
                'player_type' => $this->selectedPlayerType,
                'CharID' => $this->selectedCharacter,
                'old_character_name' => $this->selectedPlayerType === 'old' ? $this->oldCharacterName : null,
                'donation_amount' => $this->selectedPlayerType === 'donate' ? $this->donationAmount : null,
                'currency' => $this->selectedPlayerType === 'donate' ? $this->selectedCurrency : null,
                'reward_amount' => $rewardAmount,
                'old_character_proofs' => !empty($oldCharacterProofPaths) ? json_encode($oldCharacterProofPaths) : null,
                'donation_proofs' => !empty($donationProofPaths) ? json_encode($donationProofPaths) : null,
                'old_character_images' => !empty($oldCharacterImagePaths) ? json_encode($oldCharacterImagePaths) : null,
                'status' => 'pending',
                'submitted_at' => now(),
            ]);

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Application submitted successfully!',
                'duration' => 3000,
            ]);
            $this->loadUserData();

        } catch (\Illuminate\Validation\ValidationException $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Validation failed: ' . implode(', ', $e->validator->errors()->all()),
                'duration' => 5000,
            ]);
        } catch (\Exception $e) {
            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Failed to submit application: ' . $e->getMessage(),
                'duration' => 5000,
            ]);
        } finally {
            $this->isSubmitting = false;
        }
    }

    public function claimReward()
    {
        $this->isClaiming = true;

        try {
            \Log::info('Attempting to claim reward for user: ' . Auth::user()->JID);
            \Log::info('Application status: ' . $this->userApplication->status);
            \Log::info('Can be claimed: ' . ($this->userApplication->canBeClaimed() ? 'Yes' : 'No'));

            if (!$this->userApplication || !$this->userApplication->canBeClaimed()) {
                \Log::warning('Cannot claim reward - application invalid or already claimed');
                $this->dispatch('show-notification', [
                    'type' => 'error',
                    'message' => 'Unable to claim reward. Please try again.',
                    'duration' => 2500,
                ]);
                $this->isClaiming = false;
                return;
            }

            // Update user balance
            $user = Auth::user();
            \Log::info('User found: ' . ($user ? 'Yes' : 'No'));
            \Log::info('WebUser exists: ' . ($user && $user->webUser ? 'Yes' : 'No'));

            if ($user && $user->webUser) {
                $oldBalance = $user->webUser->balance;
                // Add balance to webUser's balance
                $user->webUser->balance += $this->userApplication->reward_amount;
                $user->webUser->save();

                \Log::info('Balance updated from ' . $oldBalance . ' to ' . $user->webUser->balance);
            } else {
                \Log::error('User or webUser not found');
                throw new \Exception('User account not properly configured');
            }

            // Update application status
            $this->userApplication->update([
                'status' => 'claimed',
                'claimed_at' => now(),
            ]);

            \Log::info('Application status updated to claimed');

            // Refresh the application data to reflect the changes
            $this->userApplication->refresh();

            $this->dispatch('show-notification', [
                'type' => 'success',
                'message' => 'Reward claimed successfully! Your balance has been updated.',
                'duration' => 3000,
            ]);

            $this->dispatch('updateBalances');
            $this->loadUserData();

        } catch (\Exception $e) {
            \Log::error('Error claiming reward: ' . $e->getMessage());
            \Log::error('Stack trace: ' . $e->getTraceAsString());

            $this->dispatch('show-notification', [
                'type' => 'error',
                'message' => 'Error claiming reward: ' . $e->getMessage(),
                'duration' => 2500,
            ]);
        } finally {
            $this->isClaiming = false;
        }
    }

    private function getRewardAmount()
    {
        switch ($this->selectedPlayerType) {
            case 'new':
                return 10;
            case 'old':
                return 40;
            case 'donate':
                return $this->calculatedBalance;
            default:
                return 0;
        }
    }

    public function resetApplication()
    {
        // Don't delete the existing application, just reset form fields
        $this->selectedPlayerType = null;
        $this->selectedCharacter = null;
        $this->oldCharacterName = '';
        $this->oldCharacterProofs = [];
        $this->donationAmount = 0;
        $this->selectedCurrency = 'USD';
        $this->donationProofs = [];
        $this->oldCharacterImages = [];
        $this->calculatedBalance = 0;

        // If user has a rejected application, allow them to create a new one
        if ($this->userApplication && $this->userApplication->status === 'rejected') {
            // Create a new application entry while keeping the old one for history
            $this->userApplication = null;

            $this->dispatch('show-notification', [
                'type' => 'info',
                'message' => 'Ready to create a new application. Your previous application history has been preserved.',
                'duration' => 3000,
            ]);
        }
    }

    public function with()
    {
        return [
            'hasApplication' => !is_null($this->userApplication),
        ];
    }
}; ?>

<div class="welcome-event-container" x-data="{
    selectedPlayer: @entangle('selectedPlayerType'),
    selectedCharacter: @entangle('selectedCharacter'),
    donationAmount: @entangle('donationAmount'),
    currency: @entangle('selectedCurrency'),
    oldCharacterName: @entangle('oldCharacterName'),

    // Rejection notification data
    showRejectionNotification: false,
    rejectionData: {},

    // Helper methods
    selectPlayerType(type) {
        this.selectedPlayer = type;
    },

    selectCharacterFromList(charId) {
        this.selectedCharacter = charId;
        $wire.selectCharacter(charId);
    },

    // Handle rejection notification
    showRejectionAlert(data) {
        this.rejectionData = data;
        this.showRejectionNotification = true;

        // Auto hide after 10 seconds
        setTimeout(() => {
            this.hideRejectionNotification();
        }, 10000);
    },

    hideRejectionNotification() {
        this.showRejectionNotification = false;
        setTimeout(() => {
            this.rejectionData = {};
        }, 500);
    }
}" @show-rejection-notification.window="showRejectionAlert($event.detail)">

    {{-- Rejection Notification Overlay --}}
    <div class="rejection-notification-overlay"
         x-show="showRejectionNotification"
         x-transition:enter="notification-enter"
         x-transition:enter-start="notification-enter-start"
         x-transition:enter-end="notification-enter-end"
         x-transition:leave="notification-leave"
         x-transition:leave-start="notification-leave-start"
         x-transition:leave-end="notification-leave-end"
         @click="hideRejectionNotification()"
         x-cloak>

        <div class="rejection-notification-modal" @click.stop>
            {{-- Close Button --}}
            <button class="notification-close-btn" @click="hideRejectionNotification()">
                <svg viewBox="0 0 24 24" fill="currentColor">
                    <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                </svg>
            </button>

            {{-- Notification Header --}}
            <div class="notification-header">
                <div class="notification-icon-container">
                    <div class="notification-icon rejected">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                        </svg>
                    </div>
                    <div class="notification-pulse-rings">
                        <div class="pulse-ring ring-1"></div>
                        <div class="pulse-ring ring-2"></div>
                        <div class="pulse-ring ring-3"></div>
                    </div>
                </div>

                <div class="notification-content">
                    <h2 class="notification-title" data-translate="page.4t-app.rejectionNotification.title"></h2>
                    <p class="notification-subtitle" data-translate="page.4t-app.rejectionNotification.subtitle"></p>
                </div>
            </div>

            {{-- Application Details --}}
            <div class="notification-details">
                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M11,9H13V7H11M12,20C7.59,20 4,16.41 4,12C4,7.59 7.59,4 12,4C16.41,4 20,7.59 20,12C20,16.41 16.41,20 12,20M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.applicationId"></span>
                    </div>
                    <div class="detail-value" x-text="'#' + rejectionData.applicationId"></div>
                </div>

                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12,2L13.09,8.26L18,9L13.09,9.74L12,16L10.91,9.74L6,9L10.91,8.26L12,2Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.playerType"></span>
                    </div>
                    <div class="detail-value" x-text="rejectionData.playerType"></div>
                </div>

                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M12,2A2,2 0 0,1 14,4C14,4.74 13.6,5.39 13,5.73V7H14A7,7 0 0,1 21,14H22A1,1 0 0,1 23,15V18A1,1 0 0,1 22,19H21V20A2,2 0 0,1 19,22H5A2,2 0 0,1 3,20V19H2A1,1 0 0,1 1,18V15A1,1 0 0,1 2,14H3A7,7 0 0,1 10,7H11V5.73C10.4,5.39 10,4.74 10,4A2,2 0 0,1 12,2M7.5,13A2.5,2.5 0 0,0 5,15.5A2.5,2.5 0 0,0 7.5,18A2.5,2.5 0 0,0 10,15.5A2.5,2.5 0 0,0 7.5,13M16.5,13A2.5,2.5 0 0,0 14,15.5A2.5,2.5 0 0,0 16.5,18A2.5,2.5 0 0,0 19,15.5A2.5,2.5 0 0,0 16.5,13Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.requestedReward"></span>
                    </div>
                    <div class="detail-value" x-text="rejectionData.rewardAmount + ' Balance'"></div>
                </div>

                <div class="detail-row">
                    <div class="detail-label">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M19,3H18V1H16V3H8V1H6V3H5A2,2 0 0,0 3,5V19A2,2 0 0,0 5,21H19A2,2 0 0,0 21,19V5A2,2 0 0,0 19,3M19,19H5V8H19V19Z"/>
                        </svg>
                        <span data-translate="page.4t-app.rejectionNotification.reviewedAt"></span>
                    </div>
                    <div class="detail-value" x-text="rejectionData.reviewedAt"></div>
                </div>
            </div>

            {{-- Rejection Reason --}}
            <div class="rejection-reason-section" x-show="rejectionData.rejectionReason">
                <h3 class="reason-title" data-translate="page.4t-app.rejectionNotification.rejectionReason"></h3>
                <div class="reason-content">
                    <div class="reason-icon">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M11,9H13V7H11M12,20C7.59,20 4,16.41 4,12C4,7.59 7.59,4 12,4C16.41,4 20,7.59 20,12C20,16.41 16.41,20 12,20M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M11,17H13V11H11V17Z"/>
                        </svg>
                    </div>
                    <p x-text="rejectionData.rejectionReason"></p>
                </div>
            </div>

            {{-- Action Buttons --}}
            <div class="notification-actions">
                <button class="notification-btn secondary" @click="hideRejectionNotification()" data-translate="page.4t-app.rejectionNotification.dismiss"></button>
                <button class="notification-btn primary" @click="hideRejectionNotification(); $wire.resetApplication()" data-translate="page.4t-app.rejectionNotification.tryAgain"></button>
            </div>
        </div>
    </div>
    {{-- Header Section --}}
    <div class="welcome-event-header">
        <h1 class="welcome-title" data-translate="welcomeEvent.title"></h1>
        <p class="welcome-description" data-translate="welcomeEvent.description"></p>
    </div>

    {{-- Rejected Applications Section --}}
    @php
        $rejectedApplications = $this->getRejectedApplications();
    @endphp

    @if($rejectedApplications->count() > 0)
        <div class="rejected-applications-section"
             x-data="{
                 showRejected: true,
                 selectedRejection: null,

                 showRejectionDetails(application) {
                     this.selectedRejection = application;
                 },

                 hideRejectionDetails() {
                     this.selectedRejection = null;
                 }
             }">

            <div class="rejected-section-header">
                <div class="rejected-icon">
                    <svg viewBox="0 0 24 24" fill="currentColor">
                        <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                    </svg>
                </div>
                <div class="rejected-header-content">
                    <h2 class="rejected-title" data-translate="page.4t-app.rejectedApplicationsSection.title"></h2>
                    <p class="rejected-subtitle" data-translate="page.4t-app.rejectedApplicationsSection.subtitle"></p>
                </div>
                <button class="toggle-rejected-btn" @click="showRejected = !showRejected">
                    <svg x-show="!showRejected" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M7.41,8.58L12,13.17L16.59,8.58L18,10L12,16L6,10L7.41,8.58Z"/>
                    </svg>
                    <svg x-show="showRejected" viewBox="0 0 24 24" fill="currentColor">
                        <path d="M7.41,15.41L12,10.83L16.59,15.41L18,14L12,8L6,14L7.41,15.41Z"/>
                    </svg>
                </button>
            </div>

            <div class="rejected-applications-list"
                 x-show="showRejected"
                 x-transition:enter="transition ease-out duration-300"
                 x-transition:enter-start="opacity-0 transform -translate-y-4"
                 x-transition:enter-end="opacity-100 transform translate-y-0">

                @foreach($rejectedApplications as $rejection)
                    <div class="rejected-application-card"
                         @click="showRejectionDetails({
                             id: {{ $rejection->id }},
                             playerType: '{{ ucfirst($rejection->player_type) }} Player',
                             rewardAmount: {{ $rejection->reward_amount }},
                             reviewedAt: '{{ $rejection->reviewed_at?->format('M d, Y H:i') }}',
                             rejectionReason: '{{ addslashes($rejection->rejection_reason ?? 'No specific reason provided') }}',
                             characterName: '{{ $rejection->character?->CharName16 ?? 'N/A' }}'
                         })">

                        <div class="rejection-card-header">
                            <div class="rejection-card-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                                </svg>
                            </div>
                            <div class="rejection-card-info">
                                <h4 class="rejection-card-title">
                                    <span data-translate="page.4t-app.rejectedApplicationsSection.applicationId"></span> #{{ $rejection->id }}
                                </h4>
                                <p class="rejection-card-type">{{ ucfirst($rejection->player_type) }} Player</p>
                            </div>
                            <div class="rejection-card-meta">
                                <span class="rejection-reward">{{ $rejection->reward_amount }} Balance</span>
                                <span class="rejection-date">{{ $rejection->reviewed_at?->format('M d') }}</span>
                            </div>
                        </div>

                        <div class="rejection-reason-preview">
                            <p>{{ Str::limit($rejection->rejection_reason ?? 'No specific reason provided', 80) }}</p>
                        </div>

                        <div class="rejection-card-actions">
                            <span class="view-details-btn" data-translate="page.4t-app.rejectedApplicationsSection.viewDetails"></span>
                            <button class="try-again-small-btn"
                                    @click.stop="$wire.resetApplication()"
                                    data-translate="page.4t-app.rejectedApplicationsSection.tryAgain"></button>
                        </div>
                    </div>
                @endforeach
            </div>

            {{-- Rejection Details Modal --}}
            <div class="rejection-details-overlay"
                 x-show="selectedRejection"
                 x-transition:enter="transition ease-out duration-300"
                 x-transition:enter-start="opacity-0"
                 x-transition:enter-end="opacity-100"
                 @click="hideRejectionDetails()"
                 x-cloak>

                <div class="rejection-details-modal" @click.stop>
                    <button class="modal-close-btn" @click="hideRejectionDetails()">
                        <svg viewBox="0 0 24 24" fill="currentColor">
                            <path d="M19,6.41L17.59,5L12,10.59L6.41,5L5,6.41L10.59,12L5,17.59L6.41,19L12,13.41L17.59,19L19,17.59L13.41,12L19,6.41Z"/>
                        </svg>
                    </button>

                    <div class="modal-header">
                        <div class="modal-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                            </svg>
                        </div>
                        <div class="modal-title-section">
                            <h3 data-translate="page.4t-app.rejectedApplicationsSection.applicationDetails"></h3>
                            <p x-text="'Application #' + selectedRejection?.id"></p>
                        </div>
                    </div>

                    <div class="modal-details">
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.playerType"></span>
                            <span class="detail-value" x-text="selectedRejection?.playerType"></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.requestedReward"></span>
                            <span class="detail-value" x-text="selectedRejection?.rewardAmount + ' Balance'"></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.reviewedAt"></span>
                            <span class="detail-value" x-text="selectedRejection?.reviewedAt"></span>
                        </div>
                        <div class="detail-item">
                            <span class="detail-label" data-translate="page.4t-app.rejectedApplicationsSection.characterName"></span>
                            <span class="detail-value" x-text="selectedRejection?.characterName"></span>
                        </div>
                    </div>

                    <div class="modal-reason">
                        <h4 data-translate="page.4t-app.rejectedApplicationsSection.rejectionReason"></h4>
                        <p x-text="selectedRejection?.rejectionReason"></p>
                    </div>

                    <div class="modal-actions">
                        <button class="modal-btn secondary" @click="hideRejectionDetails()" data-translate="page.4t-app.rejectedApplicationsSection.close"></button>
                        <button class="modal-btn primary" @click="hideRejectionDetails(); $wire.resetApplication()" data-translate="page.4t-app.rejectedApplicationsSection.createNewApplication"></button>
                    </div>
                </div>
            </div>
        </div>
    @endif

    {{-- Main Content --}}
    @if (!$hasApplication)
        {{-- Welcome Card Based on Player Type --}}
        @if ($selectedPlayerType)
            <div class="welcome-celebration-card {{ $selectedPlayerType }}-player"
                 x-data="{ show: false }"
                 x-init="setTimeout(() => show = true, 100)"
                 :class="{ 'show': show }">

                {{-- Floating Particles --}}
                <div class="welcome-particles">
                    @for($i = 0; $i < 8; $i++)
                        <div class="welcome-particle {{ $selectedPlayerType }}-player"
                             style="left: {{ rand(0, 100) }}%; animation-delay: {{ $i * 1.2 }}s;"></div>
                    @endfor
                </div>

                {{-- Card Header --}}
                <div class="welcome-card-header">
                    <div class="welcome-card-icon {{ $selectedPlayerType }}-player">
                        @if($selectedPlayerType === 'new')
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2L13.09 8.26L18 9L13.09 9.74L12 16L10.91 9.74L6 9L10.91 8.26L12 2Z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'old')
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M11.5 14L8.5 17L7.91 16.41L10.5 13.83L7.91 11.24L8.5 10.65L11.5 13.65L14.5 10.65L15.09 11.24L12.5 13.83L15.09 16.41L14.5 17L11.5 14ZM12 2A10 10 0 0 1 22 12A10 10 0 0 1 12 22A10 10 0 0 1 2 12A10 10 0 0 1 12 2Z"/>
                            </svg>
                        @elseif($selectedPlayerType === 'donate')
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM13 7H11V22H13V7ZM7 11H9V22H7V11ZM15 15H17V22H15V15Z"/>
                            </svg>
                        @endif
                    </div>

                    <div class="welcome-card-title-section">
                        <h2 class="welcome-card-title" data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.title"></h2>
                        <p class="welcome-card-subtitle" data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.subtitle"></p>
                    </div>
                </div>

                {{-- Card Content --}}
                <div class="welcome-card-content">
                    <p class="welcome-card-description" data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.description"></p>

                    {{-- Features Grid --}}
                    <div class="welcome-card-features">
                        @for($i = 0; $i < 4; $i++)
                            <div class="welcome-feature-item">
                                <div class="welcome-feature-icon {{ $selectedPlayerType }}-player">
                                    @if($i === 0)
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM21 9V7L15 1H5C3.89 1 3 1.89 3 3V21C3 22.11 3.89 23 5 23H19C20.11 23 21 22.11 21 21V9H21Z"/>
                                        </svg>
                                    @elseif($i === 1)
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M12 2L3.09 8.26L4 9L11 14L20 9L21.09 8.26L12 2Z"/>
                                        </svg>
                                    @elseif($i === 2)
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M16 4C18.2 4 20 5.8 20 8C20 10.2 18.2 12 16 12C13.8 12 12 10.2 12 8C12 5.8 13.8 4 16 4ZM18 14.54C19.24 15.19 20 16.45 20 18V22H22V18C22 16.81 20.7 15.6 18.64 15.16L18 14.54Z"/>
                                        </svg>
                                    @else
                                        <svg viewBox="0 0 24 24" fill="currentColor">
                                            <path d="M19 3H5C3.9 3 3 3.9 3 5V19C3 20.1 3.9 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.9 20.1 3 19 3ZM19 19H5V5H19V19ZM17 12H7V10H17V12Z"/>
                                        </svg>
                                    @endif
                                </div>
                                <span class="welcome-feature-text" data-translate="welcomeEvent.welcomeCards.{{ $selectedPlayerType }}Player.features.{{ $i }}"></span>
                            </div>
                        @endfor
                    </div>
                </div>
            </div>
        @endif

        {{-- Player Type Selection --}}
        @if (!$selectedPlayerType)
            <div class="player-type-selection">
                <h2 class="section-title" data-translate="welcomeEvent.selectPlayerType"></h2>
                <div class="player-types-grid">
                    {{-- New Player Card --}}
                    <div class="player-type-card new-player"
                         @click="$wire.selectPlayerType('new')"
                         x-transition:enter="card-enter">
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2L13.09 8.26L18 9L13.09 9.74L12 16L10.91 9.74L6 9L10.91 8.26L12 2Z"/>
                            </svg>
                        </div>
                        <h3 class="card-title" data-translate="welcomeEvent.newPlayer.title"></h3>
                        <p class="card-subtitle" data-translate="welcomeEvent.newPlayer.subtitle"></p>
                        <div class="card-reward" data-translate="welcomeEvent.newPlayer.reward"></div>
                        <p class="card-description" data-translate="welcomeEvent.newPlayer.description"></p>
                        <button class="select-type-btn" data-translate="welcomeEvent.buttons.selectType"></button>
                    </div>

                    {{-- Old Player Card --}}
                    <div class="player-type-card old-player"
                         @click="$wire.selectPlayerType('old')"
                         x-transition:enter="card-enter"
                         x-transition:enter.delay.100ms>
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12 2C13.1 2 14 2.9 14 4C14 5.1 13.1 6 12 6C10.9 6 10 5.1 10 4C10 2.9 10.9 2 12 2ZM21 9V7L15 1H5C3.89 1 3 1.89 3 3V21C3 22.11 3.89 23 5 23H19C20.11 23 21 22.11 21 21V9H21ZM17 7V3.5L20.5 7H17Z"/>
                            </svg>
                        </div>
                        <h3 class="card-title" data-translate="welcomeEvent.oldPlayer.title"></h3>
                        <p class="card-subtitle" data-translate="welcomeEvent.oldPlayer.subtitle"></p>
                        <div class="card-reward" data-translate="welcomeEvent.oldPlayer.reward"></div>
                        <p class="card-description" data-translate="welcomeEvent.oldPlayer.description"></p>
                        <button class="select-type-btn" data-translate="welcomeEvent.buttons.selectType"></button>
                    </div>

                    {{-- donate Player Card --}}
                    <div class="player-type-card donate-player"
                         @click="$wire.selectPlayerType('donate')"
                         x-transition:enter="card-enter"
                         x-transition:enter.delay.200ms>
                        <div class="card-icon">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,2A3,3 0 0,1 15,5V11A3,3 0 0,1 12,14A3,3 0 0,1 9,11V5A3,3 0 0,1 12,2M19,11C19,14.53 16.39,17.44 13,17.93V21H11V17.93C7.61,17.44 5,14.53 5,11H7A5,5 0 0,0 12,16A5,5 0 0,0 17,11H19Z"/>
                            </svg>
                        </div>
                        <h3 class="card-title" data-translate="welcomeEvent.donatePlayer.title"></h3>
                        <p class="card-subtitle" data-translate="welcomeEvent.donatePlayer.subtitle"></p>
                        <div class="card-reward" data-translate="welcomeEvent.donatePlayer.reward"></div>
                        <p class="card-description" data-translate="welcomeEvent.donatePlayer.description"></p>
                        <button class="select-type-btn" data-translate="welcomeEvent.buttons.selectType"></button>
                    </div>
                </div>
            </div>
        @else
            {{-- Application Form --}}
            <div class="application-form" x-transition:enter="form-enter">
                <h2 class="section-title">
                    @if($selectedPlayerType === 'new')
                        <span data-translate="welcomeEvent.newPlayer.title"></span>
                    @elseif($selectedPlayerType === 'old')
                        <span data-translate="welcomeEvent.oldPlayer.title"></span>
                    @elseif($selectedPlayerType === 'donate')
                        <span data-translate="welcomeEvent.donatePlayer.title"></span>
                    @endif
                    <span data-translate="welcomeEvent.forms.characterSelection"></span>
                </h2>

                {{-- Character Selection --}}
                <div class="form-section">
                    <label class="form-label" data-translate="welcomeEvent.forms.selectCharacter"></label>
                    <p class="form-help" data-translate="welcomeEvent.forms.selectCharacterDesc"></p>

                    <div class="character-selection">
                        @forelse($userCharacters as $character)
                            @php
                                $isDisabled = !$character['isActive']; // All characters must be active now
                            @endphp
                            <div class="character-option @if($selectedCharacter == $character['CharID']) selected @endif @if($isDisabled) disabled @endif"
                                 @if(!$isDisabled) wire:click="selectCharacter({{ $character['CharID'] }})" @endif>
                                <div class="character-avatar">
                                    <img src="{{ $character['avatar'] }}"
                                         alt="{{ $character['CharName16'] }}"
                                         onerror="this.src='{{ asset('assets/images/avatars/default.gif') }}'; this.onerror=null;">
                                    @if($isDisabled)
                                        <div class="character-status inactive" data-translate="welcomeEvent.status.inactive"></div>
                                    @elseif($character['isActive'])
                                        <div class="character-status active" data-translate="welcomeEvent.status.active"></div>
                                    @endif
                                </div>
                                <div class="character-info">
                                    <div class="character-name">{{ $character['CharName16'] }}</div>
                                    <div class="character-level">Level {{ $character['CurLevel'] ?? 1 }}</div>
                                    @if(!$character['isActive'])
                                        <div class="character-warning" data-translate="welcomeEvent.messages.characterMustBeActive"></div>
                                    @endif
                                </div>
                            </div>
                        @empty
                            <div class="no-characters">
                                <p data-translate="welcomeEvent.messages.noCharactersFound"></p>
                            </div>
                        @endforelse
                    </div>
                </div>

                {{-- Old Player Specific Fields --}}
                @if($selectedPlayerType === 'old')
                    <div class="form-section">
                        <label class="form-label" data-translate="welcomeEvent.forms.oldCharacterName"></label>
                        <input type="text"
                               wire:model="oldCharacterName"
                               class="form-input"
                               data-translate-placeholder="welcomeEvent.forms.oldCharacterNamePlaceholder">
                    </div>

                    <div class="form-section">
                        <label class="form-label" data-translate="welcomeEvent.forms.oldCharacterProof"></label>
                        <p class="form-help" data-translate="welcomeEvent.forms.oldCharacterProofDesc"></p>

                        <div class="file-upload-container" x-data="{
                            handleDrop(event) {
                                const files = Array.from(event.dataTransfer.files);
                                console.log('Files dropped:', files);
                            }
                        }">
                            <div class="file-upload-area"
                                 @drop.prevent="handleDrop($event)"
                                 @dragover.prevent
                                 @click="$refs.fileInput.click()">
                                <div class="upload-icon">📁</div>
                                <p data-translate="welcomeEvent.forms.dragAndDrop"></p>
                                <p class="upload-help" data-translate="welcomeEvent.forms.supportedFormats"></p>
                            </div>
                            <input type="file"
                                   x-ref="fileInput"
                                   wire:model="oldCharacterProofs"
                                   multiple
                                   accept="image/*"
                                   style="display: none;">
                        </div>

                        @if($oldCharacterProofs)
                            <div class="uploaded-files">
                                @foreach($oldCharacterProofs as $file)
                                    <div class="uploaded-file">
                                        <span>{{ $file->getClientOriginalName() }}</span>
                                        <span class="file-size">({{ number_format($file->getSize() / 1024, 1) }}KB)</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                @endif

                {{-- donate Player Specific Fields --}}
                @if($selectedPlayerType === 'donate')
                    <div class="form-section" x-data="{
                        isActive: false,
                        isCalculating: false,

                        init() {
                            this.$watch('$wire.donationAmount', (value) => {
                                if (value > 0) {
                                    this.isCalculating = true;
                                    setTimeout(() => {
                                        this.isCalculating = false;
                                    }, 500);
                                }
                            });
                        }
                    }">
                        <label class="form-label" data-translate="welcomeEvent.forms.donationAmount"></label>

                        <div class="amount-input-group"
                             :class="{ 'active': isActive }"
                             @click="isActive = true"
                             @clickaway="isActive = false">
                            <input type="number"
                                   wire:model.live="donationAmount"
                                   class="amount-input"
                                   data-translate-placeholder="welcomeEvent.forms.donationAmountPlaceholder"
                                   min="1"
                                   step="0.01"
                                   x-ref="amountInput"
                                   @focus="isActive = true"
                                   @blur="isActive = false">

                            <select wire:model.live="selectedCurrency" class="currency-select">
                                <option value="USD">💵 USD</option>
                                <option value="EGP">🇪🇬 EGP</option>
                            </select>
                        </div>

                        <div class="currency-rates">
                            <div class="rate usd">
                                <span>💵</span>
                                <span data-translate="welcomeEvent.forms.usdRate"></span>
                            </div>
                            <div class="rate egp">
                                <span>🇪🇬</span>
                                <span data-translate="welcomeEvent.forms.egpRate"></span>
                            </div>
                        </div>

                        @if($calculatedBalance > 0)
                            <div class="calculated-reward"
                                 :class="{ 'balance-calculating': isCalculating }"
                                 x-show="$wire.calculatedBalance > 0"
                                 x-transition:enter="transition ease-out duration-300"
                                 x-transition:enter-start="opacity-0 transform scale-95"
                                 x-transition:enter-end="opacity-100 transform scale-100">
                                <h4 data-translate="welcomeEvent.forms.calculatedReward"></h4>
                                <div class="balance-display"
                                     x-text="$wire.calculatedBalance + ' Balance'"
                                     x-transition:enter="transition ease-out duration-500 delay-200"
                                     x-transition:enter-start="opacity-0 transform scale-75"
                                     x-transition:enter-end="opacity-100 transform scale-100"></div>
                                <div class="balance-breakdown">
                                    <span x-text="$wire.donationAmount"></span>
                                    <span x-text="$wire.selectedCurrency"></span>
                                    <span data-translate="welcomeEvent.forms.balanceConversion"></span>
                                    <strong x-text="$wire.calculatedBalance + ' Balance'"></strong>
                                    <span data-translate="welcomeEvent.forms.donateBonus"></span>
                                </div>
                            </div>
                        @endif
                    </div>

                    <div class="form-section">
                        <label class="form-label" data-translate="welcomeEvent.forms.donationProofs"></label>
                        <p class="form-help" data-translate="welcomeEvent.forms.donationProofsDesc"></p>

                        <div class="file-upload-container" x-data="{
                            handleDrop(event) {
                                const files = Array.from(event.dataTransfer.files);
                                console.log('Files dropped:', files);
                            }
                        }">
                            <div class="file-upload-area"
                                 @drop.prevent="handleDrop($event)"
                                 @dragover.prevent
                                 @click="$refs.fileInput.click()">
                                <div class="upload-icon">💰</div>
                                <p data-translate="welcomeEvent.forms.dragAndDrop"></p>
                                <p class="upload-help" data-translate="welcomeEvent.forms.supportedFormats"></p>
                            </div>
                            <input type="file"
                                   x-ref="fileInput"
                                   wire:model="donationProofs"
                                   multiple
                                   accept="image/*"
                                   style="display: none;">
                        </div>

                        @if($donationProofs)
                            <div class="uploaded-files">
                                @foreach($donationProofs as $file)
                                    <div class="uploaded-file">
                                        <span>{{ $file->getClientOriginalName() }}</span>
                                        <span class="file-size">({{ number_format($file->getSize() / 1024, 1) }}KB)</span>
                                    </div>
                                @endforeach
                            </div>
                        @endif
                    </div>
                @endif

                {{-- Optional Character Images --}}
                <div class="form-section">
                    @if($selectedPlayerType === 'new')
                        <label class="form-label"><span data-translate="welcomeEvent.forms.characterIntroImages"></span> (Optional)</label>
                        <p class="form-help" data-translate="welcomeEvent.forms.characterIntroImagesDesc"></p>
                    @else
                        <label class="form-label"><span data-translate="welcomeEvent.forms.oldCharacterImages"></span> (Optional)</label>
                        <p class="form-help" data-translate="welcomeEvent.forms.oldCharacterImagesDesc"></p>
                    @endif

                    <div class="file-upload-container" x-data="{
                        handleDrop(event) {
                            const files = Array.from(event.dataTransfer.files);
                            console.log('Files dropped:', files);
                        }
                    }">
                        <div class="file-upload-area"
                             @drop.prevent="handleDrop($event)"
                             @dragover.prevent
                             @click="$refs.fileInput.click()">
                            <div class="upload-icon">🖼️</div>
                            <p data-translate="welcomeEvent.forms.dragAndDrop"></p>
                            <p class="upload-help" data-translate="welcomeEvent.forms.supportedFormats"></p>
                        </div>
                        <input type="file"
                               x-ref="fileInput"
                               wire:model="oldCharacterImages"
                               multiple
                               accept="image/*"
                               style="display: none;">
                    </div>

                    @if($oldCharacterImages)
                        <div class="uploaded-files">
                            @foreach($oldCharacterImages as $file)
                                <div class="uploaded-file">
                                    <span>{{ $file->getClientOriginalName() }}</span>
                                    <span class="file-size">({{ number_format($file->getSize() / 1024, 1) }}KB)</span>
                                </div>
                            @endforeach
                        </div>
                    @endif
                </div>

                {{-- Submit Button --}}
                <div class="form-actions">
                    <button wire:click="resetApplication" class="btn btn-secondary" data-translate="welcomeEvent.buttons.back">
                    </button>
                    <button wire:click="submitApplication"
                            class="btn btn-primary submit-btn"
                            :disabled="$wire.isSubmitting">
                        <span x-show="!$wire.isSubmitting" data-translate="welcomeEvent.forms.submitApplication"></span>
                        <span x-show="$wire.isSubmitting" data-translate="welcomeEvent.forms.submitting"></span>
                    </button>
                </div>
            </div>
        @endif
    @else
        {{-- Application Status Display --}}
        <div class="application-status" x-transition:enter="status-enter">
            @if($userApplication->status === 'pending')
                <div class="status-card pending"
                     x-data="{
                        show: false,
                        pulseActive: true,
                        timeRemaining: {
                            hours: 0,
                            minutes: 0,
                            seconds: 0,
                            total: 0
                        },
                        interval: null,
                        deadline: '{{ $userApplication->getReviewDeadline() }}',
                        progressPercentage: 0,

                        init() {
                            setTimeout(() => this.show = true, 100);
                            this.updateCountdown();
                            this.interval = setInterval(() => {
                                this.updateCountdown();
                            }, 1000);
                        },

                        destroy() {
                            if (this.interval) {
                                clearInterval(this.interval);
                            }
                        },

                        updateCountdown() {
                            const now = new Date().getTime();
                            const deadlineTime = new Date(this.deadline).getTime();
                            const submittedTime = new Date('{{ $userApplication->submitted_at }}').getTime();
                            const totalReviewTime = deadlineTime - submittedTime;
                            const distance = deadlineTime - now;

                            if (distance > 0) {
                                this.timeRemaining.hours = Math.floor(distance / (1000 * 60 * 60));
                                this.timeRemaining.minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
                                this.timeRemaining.seconds = Math.floor((distance % (1000 * 60)) / 1000);
                                this.timeRemaining.total = distance;

                                // Calculate progress percentage (how much time has passed)
                                this.progressPercentage = Math.max(0, Math.min(100, ((totalReviewTime - distance) / totalReviewTime) * 100));
                            } else {
                                this.timeRemaining.hours = 0;
                                this.timeRemaining.minutes = 0;
                                this.timeRemaining.seconds = 0;
                                this.timeRemaining.total = 0;
                                this.progressPercentage = 100;
                                this.pulseActive = false;
                                this.destroy();
                            }
                        }
                     }"
                     :class="{ 'show': show }"
                     x-cloak>

                    {{-- Floating Background Elements --}}
                    <div class="status-background-elements">
                        <div class="bg-element bg-element-1"></div>
                        <div class="bg-element bg-element-2"></div>
                        <div class="bg-element bg-element-3"></div>
                    </div>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon-container" :class="{ 'pulse': pulseActive }">
                            <div class="status-icon pending">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M16.2,16.2L11,13V7H12.5V12.2L17,14.9L16.2,16.2Z"/>
                                </svg>
                            </div>
                            <div class="status-progress-ring">
                                <svg viewBox="0 0 50 50">
                                    <circle cx="25" cy="25" r="20" stroke="rgba(245, 158, 11, 0.2)" stroke-width="4" fill="none"/>
                                    <circle cx="25" cy="25" r="20"
                                            stroke="#f59e0b"
                                            stroke-width="4"
                                            fill="none"
                                            stroke-dasharray="125.6"
                                            :stroke-dashoffset="125.6 - (125.6 * progressPercentage / 100)"
                                            stroke-linecap="round"
                                            style="transform: rotate(-90deg); transform-origin: center;"/>
                                </svg>
                            </div>
                        </div>

                        <div class="status-content">
                            <h2 class="status-title" data-translate="page.4t-app.pending.title"></h2>
                            <p class="status-description" data-translate="page.4t-app.pending.description"></p>
                            <div class="status-meta">
                                <span class="submitted-date">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M19,3H18V1H16V3H8V1H6V3H5A2,2 0 0,0 3,5V19A2,2 0 0,0 5,21H19A2,2 0 0,0 21,19V5A2,2 0 0,0 19,3M19,19H5V8H19V19Z"/>
                                    </svg>
                                    <span data-translate="page.4t-app.submittedOn"></span>: {{ $userApplication->submitted_at->format('M d, Y H:i') }}
                                </span>
                            </div>
                        </div>
                    </div>

                    {{-- Countdown Timer Section --}}
                    <div class="countdown-section">
                        <h3 class="countdown-title" data-translate="page.4t-app.reviewTimeRemaining"></h3>

                        <div class="countdown-container">
                            <div class="countdown-item" x-show="timeRemaining.hours > 0 || timeRemaining.total > 3600000">
                                <div class="countdown-value" x-text="String(timeRemaining.hours).padStart(2, '0')"></div>
                                <div class="countdown-label" data-translate="page.4t-app.hours"></div>
                            </div>

                            <div class="countdown-separator" x-show="timeRemaining.hours > 0 || timeRemaining.total > 3600000">:</div>

                            <div class="countdown-item">
                                <div class="countdown-value" x-text="String(timeRemaining.minutes).padStart(2, '0')"></div>
                                <div class="countdown-label" data-translate="page.4t-app.minutes"></div>
                            </div>

                            <div class="countdown-separator">:</div>

                            <div class="countdown-item">
                                <div class="countdown-value" x-text="String(timeRemaining.seconds).padStart(2, '0')"></div>
                                <div class="countdown-label" data-translate="page.4t-app.seconds"></div>
                            </div>
                        </div>

                        {{-- Progress Bar --}}
                        <div class="review-progress">
                            <div class="progress-label">
                                <span data-translate="page.4t-app.reviewProgress"></span>
                                <span class="progress-percentage" x-text="Math.round(progressPercentage) + '%'"></span>
                            </div>
                            <div class="progress-bar">
                                <div class="progress-fill" :style="'width: ' + progressPercentage + '%'"></div>
                                <div class="progress-glow" :style="'left: ' + progressPercentage + '%'"></div>
                            </div>
                        </div>
                    </div>

                    {{-- Status Information Cards --}}
                    <div class="status-info-grid">
                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2M11,7V9H13V7H11M11,11V17H13V11H11Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="page.4t-app.applicationId"></h4>
                                <p>#{{ $userApplication->id }}</p>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2A2,2 0 0,1 14,4C14,4.74 13.6,5.39 13,5.73V7H14A7,7 0 0,1 21,14H22A1,1 0 0,1 23,15V18A1,1 0 0,1 22,19H21V20A2,2 0 0,1 19,22H5A2,2 0 0,1 3,20V19H2A1,1 0 0,1 1,18V15A1,1 0 0,1 2,14H3A7,7 0 0,1 10,7H11V5.73C10.4,5.39 10,4.74 10,4A2,2 0 0,1 12,2M7.5,13A2.5,2.5 0 0,0 5,15.5A2.5,2.5 0 0,0 7.5,18A2.5,2.5 0 0,0 10,15.5A2.5,2.5 0 0,0 7.5,13M16.5,13A2.5,2.5 0 0,0 14,15.5A2.5,2.5 0 0,0 16.5,18A2.5,2.5 0 0,0 19,15.5A2.5,2.5 0 0,0 16.5,13Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="page.4t-app.expectedReward"></h4>
                                <p class="reward-amount">{{ $userApplication->reward_amount }} Balance</p>
                            </div>
                        </div>

                        <div class="info-card">
                            <div class="info-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2L13.09,8.26L18,9L13.09,9.74L12,16L10.91,9.74L6,9L10.91,8.26L12,2Z"/>
                                </svg>
                            </div>
                            <div class="info-content">
                                <h4 data-translate="page.4t-app.playerType"></h4>
                                <p class="player-type">
                                    @if($userApplication->player_type === 'new')
                                        <span data-translate="welcomeEvent.newPlayer.title"></span>
                                    @elseif($userApplication->player_type === 'old')
                                        <span data-translate="welcomeEvent.oldPlayer.title"></span>
                                    @elseif($userApplication->player_type === 'donate')
                                        <span data-translate="welcomeEvent.donatePlayer.title"></span>
                                    @endif
                                </p>
                            </div>
                        </div>
                    </div>

                    {{-- What Happens Next --}}
                    <div class="next-steps-section">
                        <h3 class="next-steps-title" data-translate="page.4t-app.whatHappensNext"></h3>
                        <div class="steps-timeline">
                            <div class="timeline-step completed">
                                <div class="step-icon">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                    </svg>
                                </div>
                                <div class="step-content">
                                    <h4 data-translate="page.4t-app.step1Title"></h4>
                                    <p data-translate="page.4t-app.step1Description"></p>
                                </div>
                            </div>

                            <div class="timeline-step active">
                                <div class="step-icon">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M16.2,16.2L11,13V7H12.5V12.2L17,14.9L16.2,16.2Z"/>
                                    </svg>
                                </div>
                                <div class="step-content">
                                    <h4 data-translate="page.4t-app.step2Title"></h4>
                                    <p data-translate="page.4t-app.step2Description"></p>
                                </div>
                            </div>

                            <div class="timeline-step pending">
                                <div class="step-icon">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M5,9V21H1V9H5M9,21A2,2 0 0,1 7,19V9C7,8.45 7.22,7.95 7.59,7.59L14.17,1L15.23,2.06C15.5,2.33 15.67,2.7 15.67,3.11L15.64,3.43L14.69,8H21C21.5,8 22,8.4 22,9C22,9.13 22,9.26 21.93,9.4L19.92,17.5C19.73,18.4 18.88,19 18,19H9M9,17H17.5L19,10H12.5L13.8,3.74L9,8.5V17Z"/>
                                    </svg>
                                </div>
                                <div class="step-content">
                                    <h4 data-translate="page.4t-app.step3Title"></h4>
                                    <p data-translate="page.4t-app.step3Description"></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

            @elseif($userApplication->status === 'approved')
                <div class="status-card approved"
                     x-data="{
                        show: false,
                        celebrating: false,

                        init() {
                            setTimeout(() => this.show = true, 100);
                            setTimeout(() => this.celebrating = true, 600);
                        }
                     }"
                     :class="{ 'show': show, 'celebrating': celebrating }"
                     x-cloak>

                    {{-- Celebration Particles --}}
                    <div class="celebration-particles">
                        @for($i = 0; $i < 12; $i++)
                            <div class="celebration-particle"
                                 style="left: {{ rand(0, 100) }}%; animation-delay: {{ $i * 0.3 }}s;"></div>
                        @endfor
                    </div>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon-container celebration">
                            <div class="status-icon approved">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                            </div>
                            <div class="success-rings">
                                <div class="success-ring ring-1"></div>
                                <div class="success-ring ring-2"></div>
                                <div class="success-ring ring-3"></div>
                            </div>
                        </div>

                        <div class="status-content">
                            <h2 class="status-title approved" data-translate="page.4t-app.approved.title"></h2>
                            <p class="status-description" data-translate="page.4t-app.approved.description"></p>
                            <div class="status-meta">
                                <span class="approved-date">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2M11,7V9H13V7H11M11,11V17H13V11H11Z"/>
                                    </svg>
                                    <span data-translate="page.4t-app.approvedOn"></span>: {{ $userApplication->reviewed_at?->format('M d, Y H:i') }}
                                </span>
                            </div>
                        </div>
                    </div>

                    {{-- Reward Display --}}
                    <div class="reward-showcase">
                        <div class="reward-container">
                            <h3 class="reward-title" data-translate="page.4t-app.yourReward"></h3>
                            <div class="reward-amount-display">
                                <div class="reward-icon">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M12,2A2,2 0 0,1 14,4C14,4.74 13.6,5.39 13,5.73V7H14A7,7 0 0,1 21,14H22A1,1 0 0,1 23,15V18A1,1 0 0,1 22,19H21V20A2,2 0 0,1 19,22H5A2,2 0 0,1 3,20V19H2A1,1 0 0,1 1,18V15A1,1 0 0,1 2,14H3A7,7 0 0,1 10,7H11V5.73C10.4,5.39 10,4.74 10,4A2,2 0 0,1 12,2M7.5,13A2.5,2.5 0 0,0 5,15.5A2.5,2.5 0 0,0 7.5,18A2.5,2.5 0 0,0 10,15.5A2.5,2.5 0 0,0 7.5,13M16.5,13A2.5,2.5 0 0,0 14,15.5A2.5,2.5 0 0,0 16.5,18A2.5,2.5 0 0,0 19,15.5A2.5,2.5 0 0,0 16.5,13Z"/>
                                    </svg>
                                </div>
                                <div class="reward-amount">{{ $userApplication->reward_amount }} Balance</div>
                                <p class="reward-note" data-translate="page.4t-app.rewardNote"></p>
                            </div>
                        </div>
                    </div>

                    {{-- Claim Button --}}
                    @if($userApplication->canBeClaimed())
                        <div class="claim-section" x-data="{ isHovering: false }">
                            <button wire:click="claimReward"
                                    class="claim-reward-btn"
                                    :disabled="$wire.isClaiming"
                                    :class="{ 'claiming': $wire.isClaiming, 'hover': isHovering }"
                                    @mouseenter="isHovering = true"
                                    @mouseleave="isHovering = false">
                                <div class="btn-content">
                                    <svg class="btn-icon" viewBox="0 0 24 24" fill="currentColor" x-show="!$wire.isClaiming">
                                        <path d="M5,9V21H1V9H5M9,21A2,2 0 0,1 7,19V9C7,8.45 7.22,7.95 7.59,7.59L14.17,1L15.23,2.06C15.5,2.33 15.67,2.7 15.67,3.11L15.64,3.43L14.69,8H21C21.5,8 22,8.4 22,9C22,9.13 22,9.26 21.93,9.4L19.92,17.5C19.73,18.4 18.88,19 18,19H9M9,17H17.5L19,10H12.5L13.8,3.74L9,8.5V17Z"/>
                                    </svg>
                                    <div class="btn-spinner" x-show="$wire.isClaiming">
                                        <div class="spinner"></div>
                                    </div>
                                    <span x-show="!$wire.isClaiming" data-translate="page.4t-app.claimReward"></span>
                                    <span x-show="$wire.isClaiming" data-translate="page.4t-app.claiming"></span>
                                </div>
                                <div class="btn-glow" x-show="!$wire.isClaiming"></div>
                            </button>
                        </div>
                    @else
                        <div class="already-claimed-section">
                            <div class="claimed-message">
                                <svg class="claimed-icon" viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <h3 data-translate="page.4t-app.alreadyClaimed">Reward Already Claimed</h3>
                                <p data-translate="page.4t-app.claimedOn">Claimed on: {{ $userApplication->claimed_at?->format('M d, Y H:i') }}</p>
                            </div>
                        </div>
                    @endif
                </div>

            @elseif($userApplication->status === 'rejected')
                <div class="status-card rejected"
                     x-data="{
                        show: false,
                        showNewApplicationOption: false,

                        init() {
                            setTimeout(() => this.show = true, 100);
                            setTimeout(() => this.showNewApplicationOption = true, 1000);
                        }
                     }"
                     :class="{ 'show': show }"
                     x-cloak>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon-container rejected">
                            <div class="status-icon rejected">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2C17.53,2 22,6.47 22,12C22,17.53 17.53,22 12,22C6.47,22 2,17.53 2,12C2,6.47 6.47,2 12,2M15.59,7L12,10.59L8.41,7L7,8.41L10.59,12L7,15.59L8.41,17L12,13.41L15.59,17L17,15.59L13.41,12L17,8.41L15.59,7Z"/>
                                </svg>
                            </div>
                            <div class="rejection-pulse"></div>
                        </div>

                        <div class="status-content">
                            <h2 class="status-title rejected" data-translate="page.4t-app.rejected.title"></h2>
                            <p class="status-description" data-translate="page.4t-app.rejected.description"></p>
                            <div class="status-meta">
                                <span class="rejected-date">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M19,3H18V1H16V3H8V1H6V3H5A2,2 0 0,0 3,5V19A2,2 0 0,0 5,21H19A2,2 0 0,0 21,19V5A2,2 0 0,0 19,3M19,19H5V8H19V19Z"/>
                                    </svg>
                                    <span data-translate="page.4t-app.rejectedOn"></span>: {{ $userApplication->reviewed_at?->format('M d, Y H:i') }}
                                </span>
                            </div>
                        </div>
                    </div>

                    {{-- Rejection Reason --}}
                    @if($userApplication->rejection_reason)
                        <div class="rejection-reason-section">
                            <h3 class="reason-title" data-translate="page.4t-app.rejectionReason"></h3>
                            <div class="reason-content">
                                <div class="reason-icon">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M11,9H13V7H11M12,20C7.59,20 4,16.41 4,12C4,7.59 7.59,4 12,4C16.41,4 20,7.59 20,12C20,16.41 16.41,20 12,20M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M11,17H13V11H11V17Z"/>
                                    </svg>
                                </div>
                                <p>{{ $userApplication->rejection_reason }}</p>
                            </div>
                        </div>
                    @endif

                    {{-- New Application Option --}}
                    <div class="new-application-section"
                         x-show="showNewApplicationOption"
                         x-transition:enter="transition ease-out duration-500"
                         x-transition:enter-start="opacity-0 transform translateY(20px)"
                         x-transition:enter-end="opacity-100 transform translateY(0)">

                        <div class="new-app-header">
                            <div class="new-app-icon">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2M11,7V9H13V7H11M11,11V17H13V11H11Z"/>
                                </svg>
                            </div>
                            <h3 data-translate="page.4t-app.newApplication.title"></h3>
                        </div>

                        <p class="new-app-description" data-translate="page.4t-app.newApplication.description"></p>

                        <div class="new-app-features">
                            <div class="feature-item">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <span data-translate="page.4t-app.newApplication.feature1"></span>
                            </div>
                            <div class="feature-item">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <span data-translate="page.4t-app.newApplication.feature2"></span>
                            </div>
                            <div class="feature-item">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <span data-translate="page.4t-app.newApplication.feature3"></span>
                            </div>
                        </div>

                        <button wire:click="resetApplication"
                                class="new-application-btn"
                                x-data="{ creating: false }"
                                @click="creating = true"
                                :class="{ 'creating': creating }">
                            <div class="btn-content">
                                <svg class="btn-icon" viewBox="0 0 24 24" fill="currentColor" x-show="!creating">
                                    <path d="M19,13H13V19H11V13H5V11H11V5H13V11H19V13Z"/>
                                </svg>
                                <div class="btn-spinner" x-show="creating">
                                    <div class="spinner"></div>
                                </div>
                                <span x-show="!creating" data-translate="page.4t-app.newApplication.createNew"></span>
                                <span x-show="creating" data-translate="page.4t-app.newApplication.creating"></span>
                            </div>
                        </button>
                    </div>
                </div>

            @elseif($userApplication->status === 'claimed')
                <div class="status-card claimed"
                     x-data="{
                        show: false,
                        confetti: false,

                        init() {
                            setTimeout(() => this.show = true, 100);
                            setTimeout(() => this.confetti = true, 300);
                        }
                     }"
                     :class="{ 'show': show, 'confetti': confetti }"
                     x-cloak>

                    {{-- Confetti Animation --}}
                    <div class="confetti-container" x-show="confetti">
                        @for($i = 0; $i < 20; $i++)
                            <div class="confetti-piece confetti-{{ $i % 4 + 1 }}"
                                 style="left: {{ rand(0, 100) }}%; animation-delay: {{ $i * 0.1 }}s;"></div>
                        @endfor
                    </div>

                    {{-- Header Section --}}
                    <div class="status-header-section">
                        <div class="status-icon-container claimed">
                            <div class="status-icon claimed">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M16,6L18.29,8.29L13.41,13.17L9.41,9.17L10.83,7.75L13.41,10.34L16,6M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2Z"/>
                                </svg>
                            </div>
                            <div class="claimed-glow"></div>
                        </div>

                        <div class="status-content">
                            <h2 class="status-title claimed" data-translate="page.4t-app.claimed.title"></h2>
                            <p class="status-description" data-translate="page.4t-app.claimed.description"></p>
                            <div class="status-meta">
                                <span class="claimed-date">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M12,2A10,10 0 0,1 22,12A10,10 0 0,1 12,22A10,10 0 0,1 2,12A10,10 0 0,1 12,2M11,7V9H13V7H11M11,11V17H13V11H11Z"/>
                                    </svg>
                                    <span data-translate="page.4t-app.claimedOn"></span>: {{ $userApplication->claimed_at?->format('M d, Y H:i') }}
                                </span>
                            </div>
                        </div>
                    </div>

                    {{-- Claimed Amount Display --}}
                    <div class="claimed-amount-section">
                        <div class="claimed-container">
                            <h3 class="claimed-title" data-translate="page.4t-app.claimedAmount"></h3>
                            <div class="claimed-amount-display">
                                <div class="claimed-badge">
                                    <svg viewBox="0 0 24 24" fill="currentColor">
                                        <path d="M5,16L3,5H1V3H4L6,14H18L20,7H8V5H21A1,1 0 0,1 22,6A1,1 0 0,1 21.8,6.6L19.3,15.6C19.1,16.2 18.7,16.5 18,16.5H6C5.3,16.5 4.9,16.2 4.7,15.6L5,16M7,18A2,2 0 0,1 9,20A2,2 0 0,1 7,22A2,2 0 0,1 5,20A2,2 0 0,1 7,18M17,18A2,2 0 0,1 19,20A2,2 0 0,1 17,22A2,2 0 0,1 15,20A2,2 0 0,1 17,18Z"/>
                                    </svg>
                                </div>
                                <div class="claimed-amount">{{ $userApplication->reward_amount }} Balance</div>
                                <p class="claimed-note" data-translate="page.4t-app.alreadyAdded"></p>
                            </div>
                        </div>

                        {{-- Success Features --}}
                        <div class="success-features">
                            <div class="success-item">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <span data-translate="page.4t-app.success.feature1"></span>
                            </div>
                            <div class="success-item">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <span data-translate="page.4t-app.success.feature2"></span>
                            </div>
                            <div class="success-item">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M21,7L9,19L3.5,13.5L4.91,12.09L9,16.17L19.59,5.59L21,7Z"/>
                                </svg>
                                <span data-translate="page.4t-app.success.feature3"></span>
                            </div>
                        </div>
                    </div>
                </div>
            @endif

            {{-- Application Details --}}
            <div class="application-details">
                <h3 data-translate="page.4t-app.applicationDetails.title"></h3>
                <div class="details-grid">
                    <div class="detail-item">
                        <div class="detail-label">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,2L13.09,8.26L18,9L13.09,9.74L12,16L10.91,9.74L6,9L10.91,8.26L12,2Z"/>
                            </svg>
                            <span class="label" data-translate="welcomeEvent.details.playerType"></span>
                        </div>
                        <span class="value">{{ __('welcomeEvent.' . $userApplication->player_type . 'Player.title') }}</span>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M12,2A2,2 0 0,1 14,4C14,4.74 13.6,5.39 13,5.73V7H14A7,7 0 0,1 21,14H22A1,1 0 0,1 23,15V18A1,1 0 0,1 22,19H21V20A2,2 0 0,1 19,22H5A2,2 0 0,1 3,20V19H2A1,1 0 0,1 1,18V15A1,1 0 0,1 2,14H3A7,7 0 0,1 10,7H11V5.73C10.4,5.39 10,4.74 10,4A2,2 0 0,1 12,2M7.5,13A2.5,2.5 0 0,0 5,15.5A2.5,2.5 0 0,0 7.5,18A2.5,2.5 0 0,0 10,15.5A2.5,2.5 0 0,0 7.5,13M16.5,13A2.5,2.5 0 0,0 14,15.5A2.5,2.5 0 0,0 16.5,18A2.5,2.5 0 0,0 19,15.5A2.5,2.5 0 0,0 16.5,13Z"/>
                            </svg>
                            <span class="label" data-translate="welcomeEvent.details.characterName"></span>
                        </div>
                        <span class="value">{{ $userApplication->character->CharName16 ?? 'N/A' }}</span>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M7,15H9C9,16.08 10.37,17 12,17C13.63,17 15,16.08 15,15C15,13.9 13.96,13.5 11.76,12.97C9.64,12.44 7,11.78 7,9C7,7.21 8.47,5.69 10.5,5.18V3H13.5V5.18C15.53,5.69 17,7.21 17,9H15C15,7.92 13.63,7 12,7C10.37,7 9,7.92 9,9C9,10.1 10.04,10.5 12.24,11.03C14.36,11.56 17,12.22 17,15C17,16.79 15.53,18.31 13.5,18.82V21H10.5V18.82C8.47,18.31 7,16.79 7,15Z"/>
                            </svg>
                            <span class="label" data-translate="welcomeEvent.details.rewardAmount"></span>
                        </div>
                        <span class="value">{{ $userApplication->reward_amount }} Balance</span>
                    </div>
                    <div class="detail-item">
                        <div class="detail-label">
                            <svg viewBox="0 0 24 24" fill="currentColor">
                                <path d="M19,3H18V1H16V3H8V1H6V3H5A2,2 0 0,0 3,5V19A2,2 0 0,0 5,21H19A2,2 0 0,0 21,19V5A2,2 0 0,0 19,3M19,19H5V8H19V19Z"/>
                            </svg>
                            <span class="label" data-translate="welcomeEvent.details.submittedAt"></span>
                        </div>
                        <span class="value">{{ $userApplication->submitted_at->format('M d, Y H:i') }}</span>
                    </div>
                    @if($userApplication->reviewed_at)
                        <div class="detail-item">
                            <div class="detail-label">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2A10,10 0 0,0 2,12A10,10 0 0,0 12,22A10,10 0 0,0 22,12A10,10 0 0,0 12,2M16.2,16.2L11,13V7H12.5V12.2L17,14.9L16.2,16.2Z"/>
                                </svg>
                                <span class="label" data-translate="welcomeEvent.details.reviewedAt"></span>
                            </div>
                            <span class="value">{{ $userApplication->reviewed_at->format('M d, Y H:i') }}</span>
                        </div>
                    @endif
                    @if($userApplication->donation_amount)
                        <div class="detail-item">
                            <div class="detail-label">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,2A2,2 0 0,1 14,4C14,4.74 13.6,5.39 13,5.73V7H14A7,7 0 0,1 21,14H22A1,1 0 0,1 23,15V18A1,1 0 0,1 22,19H21V20A2,2 0 0,1 19,22H5A2,2 0 0,1 3,20V19H2A1,1 0 0,1 1,18V15A1,1 0 0,1 2,14H3A7,7 0 0,1 10,7H11V5.73C10.4,5.39 10,4.74 10,4A2,2 0 0,1 12,2Z"/>
                                </svg>
                                <span class="label" data-translate="welcomeEvent.details.donationAmount"></span>
                            </div>
                            <span class="value">{{ $userApplication->donation_amount }} {{ $userApplication->currency }}</span>
                        </div>
                    @endif
                    @if($userApplication->old_character_name)
                        <div class="detail-item">
                            <div class="detail-label">
                                <svg viewBox="0 0 24 24" fill="currentColor">
                                    <path d="M12,4A4,4 0 0,1 16,8A4,4 0 0,1 12,12A4,4 0 0,1 8,8A4,4 0 0,1 12,4M12,14C16.42,14 20,15.79 20,18V20H4V18C4,15.79 7.58,14 12,14Z"/>
                                </svg>
                                <span class="label" data-translate="welcomeEvent.details.oldCharacterName"></span>
                            </div>
                            <span class="value">{{ $userApplication->old_character_name }}</span>
                        </div>
                    @endif
                </div>
            </div>
        </div>
    @endif
</div>


